import React, { useState, useEffect } from 'react';
import jsPDF from 'jspdf';
import autoTable from 'jspdf-autotable';
import { Quote } from 'lucide-react';

const paymentMethods = ['Account', 'EFT', 'Cash', 'Card'];
const refillTypes = [
  { type: 'R1.2', multiplier: 1.2 },
  { type: 'R1.6', multiplier: 1.6 }
];
const dispenserTypes = [
  { type: 'R27', price: 27 },
  { type: 'R32', price: 32 }
];

const motivationalQuotes = [
  "Success is not final, failure is not fatal: it is the courage to continue that counts.",
  "The only way to do great work is to love what you do.",
  "Believe you can and you're halfway there.",
  "Don't watch the clock; do what it does. Keep going.",
  "The future depends on what you do today.",
];

interface AdditionalItem {
  name: string;
  price: string;
  quantity: number;
}

interface FormData {
  address: string;
  refills: string;
  refillType: string;
  dispensers: string;
  dispenserType: string;
  additionalItems: AdditionalItem[];
  invoiceNumber: string;
  orderNumber: string;
  paymentMethod: string;
  comment: string;
}

function App() {
  const [deliveries, setDeliveries] = useState<FormData[]>([]);
  const [addresses, setAddresses] = useState<string[]>([]);
  const [newAddress, setNewAddress] = useState('');
  const [currentQuote, setCurrentQuote] = useState('');
  const [showQuote, setShowQuote] = useState(false);

  const [formData, setFormData] = useState<FormData>({
    address: '',
    refills: '',
    refillType: 'R1.2',
    dispensers: '',
    dispenserType: 'R27',
    additionalItems: [],
    invoiceNumber: '',
    orderNumber: '',
    paymentMethod: 'Account',
    comment: '',
  });

  useEffect(() => {
    const savedAddresses = localStorage.getItem('addresses');
    if (savedAddresses) {
      setAddresses(JSON.parse(savedAddresses));
    }

    const showRandomQuote = () => {
      const randomQuote = motivationalQuotes[Math.floor(Math.random() * motivationalQuotes.length)];
      setCurrentQuote(randomQuote);
      setShowQuote(true);
      setTimeout(() => setShowQuote(false), 5000);
    };

    showRandomQuote();
    const quoteInterval = setInterval(showRandomQuote, 3600000);

    return () => clearInterval(quoteInterval);
  }, []);

  const calculateRefillCost = () => {
    const quantity = parseFloat(formData.refills) || 0;
    const multiplier = refillTypes.find(r => r.type === formData.refillType)?.multiplier || 1.2;
    return quantity * multiplier;
  };

  const calculateDispenserCost = () => {
    const quantity = parseFloat(formData.dispensers) || 0;
    const price = dispenserTypes.find(d => d.type === formData.dispenserType)?.price || 27;
    return quantity * price;
  };

  const handleAddAddress = () => {
    if (newAddress && !addresses.includes(newAddress)) {
      const updatedAddresses = [...addresses, newAddress];
      setAddresses(updatedAddresses);
      localStorage.setItem('addresses', JSON.stringify(updatedAddresses));
      setNewAddress('');
    }
  };

  const handleAddItem = () => {
    setFormData({
      ...formData,
      additionalItems: [...formData.additionalItems, { name: '', price: '', quantity: 1 }]
    });
  };

  const handleItemChange = (index: number, field: keyof AdditionalItem, value: string | number) => {
    const updatedItems = [...formData.additionalItems];
    updatedItems[index] = { ...updatedItems[index], [field]: value };
    setFormData({ ...formData, additionalItems: updatedItems });
  };

  const calculateItemTotal = (item: AdditionalItem) => {
    return parseFloat(item.price || '0') * (item.quantity || 0);
  };

  const calculateCurrentDeliveryTotal = () => {
    const refillCost = calculateRefillCost();
    const dispenserCost = calculateDispenserCost();
    const itemsCost = formData.additionalItems.reduce((sum, item) => sum + calculateItemTotal(item), 0);
    return refillCost + dispenserCost + itemsCost;
  };

  const calculateGrandTotal = () => {
    return deliveries.reduce((sum, delivery) => {
      const refillCost = parseFloat(delivery.refills || '0') * 
        (refillTypes.find(r => r.type === delivery.refillType)?.multiplier || 1.2);
      const dispenserCost = parseFloat(delivery.dispensers || '0') * 
        (dispenserTypes.find(d => d.type === delivery.dispenserType)?.price || 27);
      const itemsCost = delivery.additionalItems.reduce((itemSum, item) => itemSum + calculateItemTotal(item), 0);
      return sum + refillCost + dispenserCost + itemsCost;
    }, 0);
  };

  const handleSubmitDelivery = () => {
    if (!formData.address) return;
    setDeliveries([...deliveries, formData]);
    
    const updatedAddresses = addresses.filter(addr => addr !== formData.address);
    setAddresses(updatedAddresses);
    localStorage.setItem('addresses', JSON.stringify(updatedAddresses));
    
    setFormData({
      address: '',
      refills: '',
      refillType: 'R1.2',
      dispensers: '',
      dispenserType: 'R27',
      additionalItems: [],
      invoiceNumber: '',
      orderNumber: '',
      paymentMethod: 'Account',
      comment: '',
    });
  };

  const handleExport = () => {
    try {
      const doc = new jsPDF();
      const today = new Date().toLocaleDateString();

      doc.setFontSize(16);
      doc.text('Delivery Report', 14, 20);
      doc.setFontSize(12);
      doc.text(`Date: ${today}`, 14, 30);
      doc.text(`Total Deliveries: ${deliveries.length}`, 14, 40);

      let totalRefills = { 'R1.2': 0, 'R1.6': 0 };
      let totalDispensers = { 'R27': 0, 'R32': 0 };
      let paymentTotals: Record<string, number> = {};
      let itemTotals: Record<string, { quantity: number, total: number }> = {};
      let grandTotal = 0;

      deliveries.forEach(d => {
        totalRefills[d.refillType as keyof typeof totalRefills] += parseFloat(d.refills || '0');
        totalDispensers[d.dispenserType as keyof typeof totalDispensers] += parseFloat(d.dispensers || '0');
        
        const refillCost = parseFloat(d.refills || '0') * 
          (refillTypes.find(r => r.type === d.refillType)?.multiplier || 1.2);
        const dispenserCost = parseFloat(d.dispensers || '0') * 
          (dispenserTypes.find(dt => dt.type === d.dispenserType)?.price || 27);
        const itemsCost = d.additionalItems.reduce((sum, item) => sum + calculateItemTotal(item), 0);
        const deliveryTotal = refillCost + dispenserCost + itemsCost;
        
        grandTotal += deliveryTotal;
        paymentTotals[d.paymentMethod] = (paymentTotals[d.paymentMethod] || 0) + deliveryTotal;

        d.additionalItems.forEach(item => {
          if (!itemTotals[item.name]) {
            itemTotals[item.name] = { quantity: 0, total: 0 };
          }
          itemTotals[item.name].quantity += item.quantity;
          itemTotals[item.name].total += calculateItemTotal(item);
        });
      });

      const tableData = deliveries.map(d => {
        const refillCost = parseFloat(d.refills || '0') * 
          (refillTypes.find(r => r.type === d.refillType)?.multiplier || 1.2);
        const dispenserCost = parseFloat(d.dispensers || '0') * 
          (dispenserTypes.find(dt => dt.type === d.dispenserType)?.price || 27);
        const itemsCost = d.additionalItems.reduce((sum, i) => sum + calculateItemTotal(i), 0);
        const total = refillCost + dispenserCost + itemsCost;
        
        const items = d.additionalItems.map(i => 
          `${i.name} (${i.quantity} x R${i.price})`
        ).join(', ');
        
        return [
          d.address,
          `${d.refills} (${d.refillType})`,
          `${d.dispensers} (${d.dispenserType})`,
          items,
          `R${total.toFixed(2)}`,
          d.invoiceNumber,
          d.orderNumber,
          d.paymentMethod,
          d.comment
        ];
      });

      autoTable(doc, {
        head: [['Address', 'Refills', 'Dispensers', 'Additional Items', 'Total', 'Invoice #', 'Order #', 'Payment', 'Comment']],
        body: tableData,
        startY: 50,
        styles: { fontSize: 8 },
        columnStyles: {
          0: { cellWidth: 40 },
          3: { cellWidth: 40 },
          8: { cellWidth: 30 }
        },
      });

      const finalY = (doc as any).lastAutoTable.finalY || 150;
      let currentY = finalY + 10;

      doc.setFontSize(14);
      doc.text('Summary', 14, currentY);
      currentY += 10;

      doc.setFontSize(10);
      doc.text(`Total Refills: R1.2: ${totalRefills['R1.2']}L, R1.6: ${totalRefills['R1.6']}L`, 14, currentY);
      currentY += 7;

      doc.text(`Total Dispensers: R27: ${totalDispensers['R27']}, R32: ${totalDispensers['R32']}`, 14, currentY);
      currentY += 7;

      doc.text('Payment Method Totals:', 14, currentY);
      currentY += 7;
      Object.entries(paymentTotals).forEach(([method, total]) => {
        doc.text(`${method}: R${total.toFixed(2)}`, 20, currentY);
        currentY += 7;
      });

      currentY += 5;
      doc.text('Additional Items Summary:', 14, currentY);
      currentY += 7;
      Object.entries(itemTotals).forEach(([item, data]) => {
        doc.text(`${item}: ${data.quantity} units - Total: R${data.total.toFixed(2)}`, 20, currentY);
        currentY += 7;
      });

      doc.setFontSize(12);
      doc.setFont(undefined, 'bold');
      currentY += 10;
      doc.text(`Day's Grand Total: R${grandTotal.toFixed(2)}`, 14, currentY);

      const filename = `Delivery_Report_${today.replace(/\//g, '-')}.pdf`;
      doc.save(filename);
    } catch (error) {
      console.error('Export failed:', error);
      alert('Export failed. Please check the console for details.');
    }
  };

  return (
    <div className="min-h-screen bg-gray-100 p-8">
      {showQuote && (
        <div className="fixed top-4 right-4 max-w-md bg-white p-4 rounded-lg shadow-lg border-l-4 border-blue-500 animate-fade-in">
          <div className="flex items-start gap-3">
            <Quote className="w-6 h-6 text-blue-500 flex-shrink-0 mt-1" />
            <p className="text-gray-700">{currentQuote}</p>
          </div>
        </div>
      )}

      <div className="max-w-2xl mx-auto bg-white rounded-lg shadow-md p-6">
        <div className="flex justify-between items-center mb-6">
          <h1 className="text-2xl font-bold">Delivery Log</h1>
          <div className="text-sm text-gray-600">
            <p>Completed Deliveries: {deliveries.length}</p>
            <p>Remaining Deliveries: {addresses.length}</p>
          </div>
        </div>
        
        <div className="space-y-6">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">Select Address:</label>
              <select 
                className="w-full p-2 border rounded-md"
                value={formData.address}
                onChange={e => setFormData({ ...formData, address: e.target.value })}
              >
                <option value="">Select an address</option>
                {addresses.map(addr => (
                  <option key={addr} value={addr}>{addr}</option>
                ))}
              </select>
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">Add New Address:</label>
              <div className="flex gap-2">
                <input 
                  className="flex-1 p-2 border rounded-md"
                  value={newAddress}
                  onChange={e => setNewAddress(e.target.value)}
                  placeholder="Enter new address"
                />
                <button 
                  onClick={handleAddAddress}
                  className="px-4 py-2 bg-green-500 text-white rounded-md hover:bg-green-600"
                >
                  Add
                </button>
              </div>
            </div>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Refills (x{refillTypes.find(r => r.type === formData.refillType)?.multiplier}):
              </label>
              <div className="flex gap-2">
                <input 
                  type="number" 
                  className="flex-1 p-2 border rounded-md"
                  value={formData.refills} 
                  onChange={e => setFormData({ ...formData, refills: e.target.value })} 
                />
                <select
                  className="w-24 p-2 border rounded-md"
                  value={formData.refillType}
                  onChange={e => setFormData({ ...formData, refillType: e.target.value })}
                >
                  {refillTypes.map(type => (
                    <option key={type.type} value={type.type}>{type.type}</option>
                  ))}
                </select>
                <div className="flex items-center ml-2">
                  <span className="text-gray-700">
                    = R{calculateRefillCost().toFixed(2)}
                  </span>
                </div>
              </div>
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Dispensers (R{dispenserTypes.find(d => d.type === formData.dispenserType)?.price}/unit):
              </label>
              <div className="flex gap-2">
                <input 
                  type="number" 
                  className="flex-1 p-2 border rounded-md"
                  value={formData.dispensers} 
                  onChange={e => setFormData({ ...formData, dispensers: e.target.value })} 
                />
                <select
                  className="w-24 p-2 border rounded-md"
                  value={formData.dispenserType}
                  onChange={e => setFormData({ ...formData, dispenserType: e.target.value })}
                >
                  {dispenserTypes.map(type => (
                    <option key={type.type} value={type.type}>{type.type}</option>
                  ))}
                </select>
                <div className="flex items-center ml-2">
                  <span className="text-gray-700">
                    = R{calculateDispenserCost().toFixed(2)}
                  </span>
                </div>
              </div>
            </div>
          </div>

          <div>
            <h3 className="text-lg font-medium mb-2">Additional Items</h3>
            {formData.additionalItems.map((item, i) => (
              <div key={i} className="grid grid-cols-3 gap-4 mb-2">
                <input 
                  placeholder="Item" 
                  className="p-2 border rounded-md"
                  value={item.name} 
                  onChange={e => handleItemChange(i, 'name', e.target.value)} 
                />
                <input 
                  type="number" 
                  placeholder="Price" 
                  className="p-2 border rounded-md"
                  value={item.price} 
                  onChange={e => handleItemChange(i, 'price', e.target.value)} 
                />
                <div className="flex gap-2">
                  <input 
                    type="number" 
                    placeholder="Qty" 
                    className="p-2 border rounded-md w-full"
                    value={item.quantity} 
                    onChange={e => handleItemChange(i, 'quantity', parseInt(e.target.value) || 0)} 
                  />
                  <div className="flex items-center">
                    <span className="text-gray-700">
                      = R{calculateItemTotal(item).toFixed(2)}
                    </span>
                  </div>
                </div>
              </div>
            ))}
            <button 
              onClick={handleAddItem}
              className="mt-2 px-4 py-2 bg-blue-500 text-white rounded-md hover:bg-blue-600"
            >
              Add Item
            </button>
          </div>

          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">Invoice Number:</label>
              <input 
                className="w-full p-2 border rounded-md"
                value={formData.invoiceNumber} 
                onChange={e => setFormData({ ...formData, invoiceNumber: e.target.value })} 
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">Order Number:</label>
              <input 
                className="w-full p-2 border rounded-md"
                value={formData.orderNumber} 
                onChange={e => setFormData({ ...formData, orderNumber: e.target.value })} 
              />
            </div>
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">Payment Method:</label>
            <select 
              className="w-full p-2 border rounded-md"
              value={formData.paymentMethod} 
              onChange={e => setFormData({ ...formData, paymentMethod: e.target.value })}
            >
              {paymentMethods.map(method => <option key={method}>{method}</option>)}
            </select>
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">Comment:</label>
            <textarea 
              className="w-full p-2 border rounded-md"
              value={formData.comment} 
              onChange={e => setFormData({ ...formData, comment: e.target.value })} 
            />
          </div>

          <div className="bg-gray-50 p-4 rounded-lg">
            <h3 className="text-lg font-medium mb-2">Delivery Summary</h3>
            <div className="space-y-2">
              <p className="text-gray-700">Refills Cost: R{calculateRefillCost().toFixed(2)}</p>
              <p className="text-gray-700">Dispensers Cost: R{calculateDispenserCost().toFixed(2)}</p>
              <p className="text-gray-700">Additional Items: R{formData.additionalItems.reduce((sum, item) => sum + calculateItemTotal(item), 0).toFixed(2)}</p>
              <p className="font-medium text-gray-800">Current Delivery Total: R{calculateCurrentDeliveryTotal().toFixed(2)}</p>
              <p className="font-medium text-gray-800">Day's Grand Total: R{calculateGrandTotal().toFixed(2)}</p>
            </div>
          </div>

          <div className="flex gap-4 mt-6">
            <button 
              onClick={handleSubmitDelivery}
              disabled={!formData.address}
              className={`px-6 py-2 text-white rounded-md ${
                formData.address 
                  ? 'bg-green-500 hover:bg-green-600' 
                  : 'bg-gray-400 cursor-not-allowed'
              }`}
            >
              Submit Delivery
            </button>
            <button 
              onClick={handleExport}
              className="px-6 py-2 bg-blue-500 text-white rounded-md hover:bg-blue-600"
            >
              Export Day Summary
            </button>
          </div>
        </div>
      </div>
    </div>
  );
}

export default App;